/**
 * @file libservicelog.h
 * Header file for servicelog
 *
 * Copyright (C) 2005 IBM Corporation
 *
 * @author Nathan Fontenot <nfont@austin.ibm.com>
 * @author Michael Strosaker <strosake@us.ibm.com>
 */

#ifndef _LIBSERVICELOG_H
#define _LIBSERVICELOG_H

#include <time.h>
#include <stdint.h>
#include "db.h"

/* Forward declaration of the servicelog structure which is defined in
 * libservicelog_priv.h (included below)
 */
struct servicelog;

/**
 * @def SERVICELOG_PATH
 * Default path to the servicelog database
 */
#define SERVICELOG_PATH		"/etc/servicelog"

/**
 * @def SERVICELOG_DB_NAME
 * Default name of the servicelog database
 */
#define SERVICELOG_DB_NAME	"servicelog.db"

/**
 * @def NOTIFY_DB_NAME
 * Default name of the servicelog notification database
 */
#define NOTIFY_DB_NAME		"slog_notify.db"

/**
 * @struct sl_header
 * 
 * Structure to define the type of event data being stored in the database.
 * This structure must appear as the first thing in all event structures
 * defined for use with the servicelog database.
 */
struct sl_header {
	struct sl_header *next;
	uint32_t	db_key;		/**< db entry key */
	uint32_t	event_type;	/**< event type */
	uint32_t	version;	/**< version of the event_type */
	uint32_t	event_length;	/**< total event length */
	time_t		time_event;	/**< timestamp of event occurence */
	time_t		time_log;	/**< timestamp of event logging */
	uint32_t	severity;	/**< int field of event severity */

	uint32_t	repair_action:1;
	uint32_t	serviceable_event:1;
	uint32_t	event_repaired:1;
	uint32_t	/* reserved */ :29;
};

/* defines for sl_header.event_type */
#define SL_TYPE_OS		1
#define SL_TYPE_APP		2
#define SL_TYPE_PPC64_RTAS	3
#define SL_TYPE_PPC64_ENCL	4
#define SL_MAX_EVENT_TYPE	5	/* this should be the highest event
					 * type number + 1.
					 */

/* defines for sl_header.severity */
#define SL_SEV_FATAL		7
#define SL_SEV_ERROR		6
#define SL_SEV_ERROR_LOCAL	5
#define SL_SEV_WARNING		4
#define SL_SEV_EVENT		3
#define SL_SEV_INFO		2
#define SL_SEV_DEBUG		1

/**
 * @struct sl_query
 *
 * Structure to contain all the elements used in querying the database
 */
struct sl_query {
	int		num_types;
	uint32_t	*event_types;
	uint32_t	severity;
	uint32_t	is_serviceable;
	uint32_t	is_repair_action;
	uint32_t	is_repaired;
	time_t		start_time;
	time_t		end_time;
	struct sl_header *result;
};

#define SL_QUERY_ALL	0
#define SL_QUERY_YES	1
#define SL_QUERY_NO	2

/**
 * struct sl_event_aid
 *
 * structure containing function pointers for help in adding events to the
 * database.
 */
struct sl_event_data {
	char *title;
	int (*serv_event_add)(struct servicelog *, char *);
	int (*serv_event_remove)(struct servicelog *, char *);
	int (*repair_add)(struct servicelog *, char *);
	int (*repair_remove)(struct servicelog *, char *);
	char *(*user_to_db)(struct servicelog *, char *);
	char *(*db_to_user)(struct servicelog *, char *);
};

/**
 * sl_event_aids
 *
 * Array of sl_event_aid structs for each event type.  Please note: the 
 * elements are indexed by their event type, any event types added to the
 * sl_header need to have a corresponding entry here.
 */
extern struct sl_event_data event_fcns[SL_MAX_EVENT_TYPE]; 

/**
 * sl_repair
 * 
 * Repair Action
 */
#define SL_REPAIR_VERSION	1

struct sl_repair {
	struct sl_header head;
	char		*location;
	char		*procedure;
	uint32_t	*repairs;
	int		num_repairs;
};

/**
 * sl_os
 *
 * OS Generic.
 * Note: If there is a message string for this event it should be appended
 * after this struct itself as a NULL terminated string.  The message_length
 * field should the include the length of the string (including the NULL).
 */
#define SL_OS_VERSION		1

struct sl_os {
	struct sl_header head;
	char		refcode[9];
	char		subsystem[32];
	char		*message;
	uint32_t	repair_key;
};

/**
 * sl_app
 *
 * Application generic.
 * Note: If there is a message string for this event it should be appended
 * after this struct itself as a NULL terminated string.  The message_length
 * field should the include the length of the string (including the NULL).
 */
#define SL_APP_VERSION		1

struct sl_app {
	struct sl_header head;
	char		refcode[9];
	char		*command;
	char		*message;
	pid_t		pid;
	uint32_t	repair_key;
};

/**
 * sl_ppc64_callout
 *
 * Structure to contain data that may appear as part of ppc64_* events
 */
struct sl_ppc64_callout {
	struct sl_ppc64_callout *next;
	char		priority;
	uint32_t	type;
	uint32_t	repair_key;
	char		procedure_id[32];
	char		location[128];
	char		fru[32];
	char		serial[32];
	char		ccin[32];
};

/**
 * sl_ppc64_rtas
 *
 * PPC64 RTAS Event
 * This event has several items that may appear after the end of the
 * structure itself.  If there are any fru callouts they should appear
 * first, followed by the actual raw RTAS event itself.
 */
#define SL_PPC64_RTAS_VERSION	1

struct sl_ppc64_rtas {
	struct sl_header head;
	uint16_t	flags;
#define PPC64_RTAS_FLAGS_SCANLOG_DUMP	0x0001

	uint16_t	rtas_event_type;
	uint32_t	kernel_id;
	char		refcode[9];
	uint32_t	addl_words[8];
	char		machine_type[9];
	char		machine_serial_no[13];
	char		*description;
	struct sl_ppc64_callout *callouts;
	uint32_t	rtas_event_len;
	char		*rtas_event;
};


/**
 * sl_ppc64_encl
 *
 * PPC64 Enclosure event
 * This is meant to cover all events generated from an enclosure on a PPC64
 * machine.
 */
#define SL_PPC64_ENCL_VERSION	1

struct sl_ppc64_encl {
	struct sl_header head;
	char		*description;
	struct sl_ppc64_callout *callouts;
	char		machine_type[9];
	char		machine_serial_no[13];
	uint32_t	event_len;
	char		*event;
};

/*
 * sl_notify
 *
 * Used in the notify database to indicate when an application wants to be
 * notified of the occurrence.  The actual command (and its associated
 * command-line args) should appear as a NULL-terminated string immediately
 * after this struct.  The command_length field should be the length of this
 * string (including the NULL).
 */
#define SL_NOTIFY_VERSION	1

struct sl_notify {
	struct sl_notify *next;
	uint32_t	key;		/**< key for this record */
	uint32_t	version;	/**< version of this structure */
	time_t		created;	/**< timestamp of record creation */
	uint64_t	event_types;	/**< bitmask of event types */
	uint32_t	severity;	/**< minimum event severity */
	uint32_t	repair_action;	/**< notify of repair actions? */
	uint32_t	serviceable_event; /**< notify of serv events only? */
	uint32_t	method;		/**< notification method */
#define SL_NOTIFY_NUM_VIA_STDIN		0
#define SL_NOTIFY_NUM_VIA_CMD_LINE	1
#define SL_NOTIFY_TEXT_VIA_STDIN	2

	uint32_t	command_length;	/**< length of the command */
};

/**
 * servicelog
 * 
 * The main structure for servicelog access is done via the servicelog
 * structure.  Users shouldn't access to any of the fields in this
 * struct to use the database but instead go through the provided interfaces.
 */
struct servicelog {
	DB_ENV	    *environ;	  /**< DB environment */
	DB	    *log;	  /**< primary servicelog DB */
	DB	    *index_type;  /**< secondary DB indexed on event_type */
	DB	    *notify;	  /**< notification DB */
	int	    semaphore;	  /**< semaphore for database access */
	char	    log_path[128];/**< path to the actual database files */
	char	    err_buf[1024];/**< error buffer */
};

int servicelog_open(struct servicelog *, const char *, int);
#define SL_CREATE	1

void servicelog_close(struct servicelog *);
char *servicelog_error(struct servicelog *);
int servicelog_sync(struct servicelog *);

int servicelog_log_event(struct servicelog *, void *, uint32_t *);
int servicelog_get_event(struct servicelog *, uint32_t, void **, size_t *);
int servicelog_delete_event(struct servicelog *, uint32_t);
int servicelog_update_event(struct servicelog *, void *);

/* Servicelog Queries */
int servicelog_query(struct servicelog *, struct sl_query *);
int servicelog_query_close(struct servicelog *, struct sl_query *);

/* Notification Registration */
int servicelog_notify_query(struct servicelog *, char *, struct sl_notify **,
	int *);
int servicelog_notify_get(struct servicelog *, uint32_t, struct sl_notify **);
int servicelog_notify_add(struct servicelog *, struct sl_notify *, uint32_t *);
int servicelog_notify_update(struct servicelog *, uint32_t, struct sl_notify *);
int servicelog_notify_remove(struct servicelog *, uint32_t);
int servicelog_notify_free_list(struct sl_notify *);

/* Print Routines */
int servicelog_print_header(FILE *, struct sl_header *);
int servicelog_print_event(FILE *, void *, int);
int servicelog_print_notification_tool(FILE *, struct sl_notify *);

#endif
